// DAZ Store Hider — page_script.js (Runs in page context)
(function() {
  'use strict';

  console.log("[DAZ Hider] 🎬 Page script loaded");

  // State
  let state = {
    masterEnabled: true,
    rules: [],
    highlightRules: [],
    showOnlyRules: [],
    priceRules: { min: "", max: "", minEnabled: false, maxEnabled: false },
    colHide: true,
    colHL: true,
    colShow: true,
    colPrice: true
  };

  let isFiltering = false;
  let dazCatalogHooked = false;
  let dazFilterHooked = false;
  let filterTimeout = null;

  // Listen for state updates from content script
  window.addEventListener('message', (event) => {
    if (event.source !== window) return;

    if (event.data.type === 'DAZ_HIDER_STATE_UPDATE') {
      console.log("[DAZ Hider] 📥 State received in page script");
      state = event.data.state;

      // Update DAZ-level price filter
      updateDazPriceFilter();

      // Apply DOM-level filters
      applyFilters();
    }

    if (event.data.type === 'DAZ_HIDER_MANUAL_REFRESH') {
      console.log("[DAZ Hider] 🔄 Manual refresh requested");
      triggerDazRefresh();
    }
  });

  // Get product title
  function getProductTitle(product) {
    const selectors = [
      "[itemprop='name']", ".slab-link-title", "a.slab-link-title",
      ".product-title", ".card-title", ".title", "h3", "h2", "h1",
      "a:not(.slab-link-img):not([href^='#'])"
    ];

    for (const sel of selectors) {
      const elem = product.querySelector(sel);
      if (elem) {
        return elem.textContent.trim().toLowerCase();
      }
    }
    return "";
  }

  // Get product price
  function getProductPrice(product) {
    // Try to find the sale price first (usually the first/larger price shown)
    const priceContainer = product.querySelector('.prices-disp, .prices');

    if (priceContainer) {
      // Get all text and find the first price (which is usually the sale price)
      const text = priceContainer.textContent;
      const matches = text.match(/\$[\d,]+\.?\d*/g);

      if (matches && matches.length > 0) {
        // First match is usually the sale price
        const priceText = matches[0].replace('$', '').replace(/,/g, '');
        const price = parseFloat(priceText);
        if (!isNaN(price)) {
          return price;
        }
      }
    }

    // Fallback to old selectors
    const selectors = ['.price', '.price-final', "[itemprop='price']", '.product-price', '.base_price'];

    for (const sel of selectors) {
      const elem = product.querySelector(sel);
      if (elem) {
        const text = elem.textContent.trim() || elem.getAttribute('content');
        const match = text.match(/[\d,]+\.?\d*/);
        if (match) {
          return parseFloat(match[0].replace(/,/g, ''));
        }
      }
    }

    return null;
  }

  // Calculate how many products are filtered by price from the entire catalog
  function calculatePriceFilteredCount() {
    if (!window.dazPricing || !window.daz || !window.daz.catalog) {
      console.log("[DAZ Hider] ⚠️ Cannot calculate price filtered count - missing data");
      return 0;
    }

    // Check if price filtering is active
    const hasMaxPrice = window.dazHiderMaxPrice !== undefined && window.dazHiderMaxPrice !== null;
    const hasMinPrice = window.dazHiderMinPrice !== undefined && window.dazHiderMinPrice !== null;

    if (!hasMaxPrice && !hasMinPrice) {
      return 0;
    }

    const catalog = window.daz.catalog;

    // Use productCount (total) minus productList.length (filtered) to get hidden count
    // But only if we have price filters active
    if (catalog.productCount && catalog.productList) {
      const totalProducts = catalog.productCount;
      const visibleProducts = catalog.productList.length;
      const filteredByDaz = totalProducts - visibleProducts;

      console.log(`[DAZ Hider] 📊 Price filter count: ${totalProducts} total - ${visibleProducts} visible = ${filteredByDaz} filtered by DAZ`);

      // This includes ALL DAZ filters, but if we only have price filters active, this is accurate
      // If other DAZ filters are active, this might overcount, but it's the best we can do
      return filteredByDaz;
    }

    return 0;
  }

  // Apply DOM-level filters (HIDE/HIGHLIGHT/SHOW ONLY rules)
  function applyFilters() {
    if (isFiltering) return;
    isFiltering = true;

    const products = document.querySelectorAll('li[id^="product-"]');

    let hiddenCount = 0;
    let highlightedCount = 0;

    products.forEach(product => {
      const title = getProductTitle(product);
      const price = getProductPrice(product);

      let shouldHide = false;
      let shouldHighlight = false;
      let shouldShow = true;

      // HIDE rules
      if (state.masterEnabled && state.colHide) {
        for (const rule of state.rules) {
          if (rule.enabled && rule.text && title.includes(rule.text.toLowerCase())) {
            shouldHide = true;
            break;
          }
        }
      }

      // HIGHLIGHT rules
      if (state.masterEnabled && state.colHL) {
        for (const rule of state.highlightRules) {
          if (rule.enabled && rule.text && title.includes(rule.text.toLowerCase())) {
            shouldHighlight = true;
            break;
          }
        }
      }

      // SHOW ONLY rules
      if (state.masterEnabled && state.colShow && state.showOnlyRules.some(r => r.enabled)) {
        shouldShow = false;
        for (const rule of state.showOnlyRules) {
          if (rule.enabled && rule.text && title.includes(rule.text.toLowerCase())) {
            shouldShow = true;
            break;
          }
        }
      }

      // Apply visibility
      if (shouldHide || !shouldShow) {
        product.style.display = 'none';
        hiddenCount++;
      } else {
        product.style.display = '';
      }

      // Apply highlight
      if (shouldHighlight && !shouldHide && shouldShow) {
        product.style.border = '3px solid #00ff00';
        product.style.boxShadow = '0 0 10px #00ff00';
        highlightedCount++;
      } else {
        product.style.border = '';
        product.style.boxShadow = '';
      }
    });

    // Calculate price-filtered count from the catalog difference
    const priceFilteredCount = calculatePriceFilteredCount();
    const totalHidden = hiddenCount + priceFilteredCount;

    console.log(`[DAZ Hider] ✅ Filters applied - Hidden: ${hiddenCount} (DOM) + ${priceFilteredCount} (Price) = ${totalHidden} total, Highlighted: ${highlightedCount}`);

    // Send counts to content script
    window.postMessage({
      type: 'DAZ_HIDER_UPDATE_COUNTS',
      counts: {
        hidden: totalHidden,
        highlighted: highlightedCount,
        shown: products.length - hiddenCount
      }
    }, '*');

    isFiltering = false;
  }

  // Clear filters
  function clearFilters() {
    const products = document.querySelectorAll('li[id^="product-"]');
    products.forEach(product => {
      product.style.display = '';
      product.style.border = '';
      product.style.boxShadow = '';
    });
  }

  // ============================================================================
  // DAZ-LEVEL PRICE FILTERING (Instant, no placeholders!)
  // ============================================================================

  function hookDazFilter() {
    if (dazFilterHooked) return;
    if (!window.daz || !window.daz.filter || !window.daz.filter.filterProducts) {
      console.log("[DAZ Hider] ⚠️ daz.filter.filterProducts not available yet");
      return;
    }

    // Check if dazPricing is available
    if (!window.dazPricing) {
      console.log("[DAZ Hider] ⚠️ window.dazPricing not available yet");
      return;
    }

    // Save original filterProducts function
    if (!window.daz.filter._originalFilterProducts) {
      window.daz.filter._originalFilterProducts = window.daz.filter.filterProducts;
      console.log("[DAZ Hider] ✅ Saved original daz.filter.filterProducts");
    }

    // Hook filterProducts to add price filtering using dazPricing (the correct price source!)
    window.daz.filter.filterProducts = function(productList, filterSettings, force = false) {
      // Call original first
      let filtered = this._originalFilterProducts(productList, filterSettings, force);

      const beforeCount = filtered.length;

      // Apply max price filter if enabled
      if (window.dazHiderMaxPrice !== undefined && window.dazHiderMaxPrice !== null) {
        const maxPriceCents = window.dazHiderMaxPrice;

        filtered = filtered.filter(productId => {
          const price = window.dazPricing[productId];
          return price === undefined || price === null || price <= maxPriceCents;
        });

        console.log(`[DAZ Hider] 💰 Max price filter: ${beforeCount} → ${filtered.length} (max $${(maxPriceCents/100).toFixed(2)})`);
      }

      // Apply min price filter if enabled
      if (window.dazHiderMinPrice !== undefined && window.dazHiderMinPrice !== null) {
        const minPriceCents = window.dazHiderMinPrice;
        const beforeMinCount = filtered.length;

        filtered = filtered.filter(productId => {
          const price = window.dazPricing[productId];
          return price === undefined || price === null || price >= minPriceCents;
        });

        console.log(`[DAZ Hider] 💰 Min price filter: ${beforeMinCount} → ${filtered.length} (min $${(minPriceCents/100).toFixed(2)})`);
      }

      return filtered;
    };

    dazFilterHooked = true;
    console.log("[DAZ Hider] ✅ Hooked into daz.filter.filterProducts using dazPricing (correct price source!)");
  }

let lastPriceFilterState = null;

function updateDazPriceFilter() {
  let maxPrice = null;
  let minPrice = null;

  // Update global price filter variables
  if (state.masterEnabled && state.colPrice) {
    if (state.priceRules.maxEnabled && state.priceRules.max) {
      const maxPriceValue = parseFloat(state.priceRules.max);
      if (!isNaN(maxPriceValue)) {
        maxPrice = Math.round(maxPriceValue * 100); // Convert to cents
        window.dazHiderMaxPrice = maxPrice;
        console.log(`[DAZ Hider] 💰 Set max price filter: $${maxPriceValue}`);
      } else {
        delete window.dazHiderMaxPrice;
      }
    } else {
      delete window.dazHiderMaxPrice;
    }

    if (state.priceRules.minEnabled && state.priceRules.min) {
      const minPriceValue = parseFloat(state.priceRules.min);
      if (!isNaN(minPriceValue)) {
        minPrice = Math.round(minPriceValue * 100); // Convert to cents
        window.dazHiderMinPrice = minPrice;
        console.log(`[DAZ Hider] 💰 Set min price filter: $${minPriceValue}`);
      } else {
        delete window.dazHiderMinPrice;
      }
    } else {
      delete window.dazHiderMinPrice;
    }
  } else {
    delete window.dazHiderMaxPrice;
    delete window.dazHiderMinPrice;
  }

  // Check if price filter actually changed
  const currentPriceState = JSON.stringify({ max: maxPrice, min: minPrice });
  
  if (currentPriceState === lastPriceFilterState) {
    console.log("[DAZ Hider] ⏭️ Price filter unchanged, skipping DAZ refresh");
    return;
  }
  
  lastPriceFilterState = currentPriceState;

  // Trigger DAZ refresh ONLY if price filter changed
  if (window.daz && window.daz.catalog && typeof window.daz.catalog.applyFilters === 'function') {
    console.log("[DAZ Hider] 🔄 Triggering DAZ catalog refresh with new price filter");
    window.daz.catalog.applyFilters(true);
  }
}

  // Hook into DAZ's catalog system
  function hookDazCatalog() {
    if (dazCatalogHooked) return;

    const original = window.daz.catalog.applyFilters;
    window.daz.catalog.applyFilters = function(...args) {
      console.log("[DAZ Hider] 🔄 DAZ catalog.applyFilters() called");
      const result = original.apply(this, args);

      // Apply DOM-level filters after DAZ finishes
      setTimeout(() => {
        if (!isFiltering) {
          console.log("[DAZ Hider] 🔄 Applying DOM filters after DAZ refresh");
          applyFilters();
        }
      }, 500);

      return result;
    };

    dazCatalogHooked = true;
    console.log("[DAZ Hider] ✅ Hooked into DAZ catalog.applyFilters()");
  }

  // Trigger extension filter refresh (no DAZ interaction)
  function triggerDazRefresh() {
    console.log("[DAZ Hider] 🔄 Refreshing extension filters");
    clearFilters();
    setTimeout(() => {
      applyFilters();
    }, 100);
  }

  // Wait for DAZ catalog to be available
  function waitForDazCatalog() {
    if (window.daz && window.daz.catalog && typeof window.daz.catalog.applyFilters === 'function') {
      console.log("[DAZ Hider] ✅ window.daz.catalog found!");
      hookDazCatalog();

      // Also hook the filter system
      if (window.daz.filter && window.daz.filter.filterProducts && window.dazPricing) {
        hookDazFilter();
        updateDazPriceFilter(); // Apply initial price filter
      } else {
        // Wait for filter system and dazPricing
        setTimeout(() => {
          if (window.daz.filter && window.daz.filter.filterProducts && window.dazPricing) {
            hookDazFilter();
            updateDazPriceFilter();
          }
        }, 1000);
      }
    } else {
      setTimeout(waitForDazCatalog, 500);
    }
  }

  // Setup MutationObserver
  function setupObserver() {
    const observer = new MutationObserver((mutations) => {
      if (isFiltering) return;

      let shouldReapply = false;

      for (const mutation of mutations) {
        if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
          for (const node of mutation.addedNodes) {
            if (node.nodeType === 1 && (
              node.matches && node.matches('li[id^="product-"]') ||
              node.querySelector && node.querySelector('li[id^="product-"]')
            )) {
              shouldReapply = true;
              break;
            }
          }
        }
        if (shouldReapply) break;
      }

      if (shouldReapply) {
        clearTimeout(filterTimeout);
        filterTimeout = setTimeout(() => {
          console.log("[DAZ Hider] 🔄 Products added, reapplying DOM filters");
          applyFilters();
        }, 300);
      }
    });

    // Watch the slabs-container specifically
    const slabsContainer = document.getElementById('slabs-container');
    if (slabsContainer) {
      observer.observe(slabsContainer, {
        childList: true,
        subtree: true
      });
      console.log("[DAZ Hider] ✅ Watching #slabs-container for changes");
    } else {
      // Fallback to body if container not found yet
      observer.observe(document.body, {
        childList: true,
        subtree: true
      });
      console.log("[DAZ Hider] ⚠️ Watching document.body (slabs-container not found)");
    }
  }

  // Initialize
  waitForDazCatalog();
  setupObserver();

  console.log("[DAZ Hider] ✅ Page script initialized");
})();
